'use strict';

/* --------------------------------------------------------------
 categories_table_controller.js 2020-04-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Orders Table Controller
 *
 * This controller contains the mapping logic of the categories/articles table.
 *
 * @module Compatibility/categories_table_controller
 */
gx.compatibility.module('categories_table_controller', [gx.source + '/libs/button_dropdown'],

/**  @lends module:Compatibility/categories_table_controller */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Selector
     *
     * @var {object}
     */
    $this = $(this),


    /**
     * Default Options
     *
     * @type {object}
     */
    defaults = {},


    /**
     * Final Options
     *
     * @var {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {};

    // ------------------------------------------------------------------------
    // PRIVATE METHODS
    // ------------------------------------------------------------------------

    /**
     * Get Url Parameter
     *
     * Gets a specific URL get parameter from the address bar,
     * which name should be provided as an argument.
     * @param {string} parameterName
     * @returns {object}
     * @private
     */
    var _getUrlParameter = function _getUrlParameter(parameterName) {
        var results = new RegExp('[\?&]' + parameterName + '=([^&#]*)').exec(window.location.href);
        if (results == null) {
            return null;
        } else {
            return results[1] || 0;
        }
    };

    /**
     * Product ID
     *
     * Holds the product id from the get parameter.
     * @type {object}
     */
    var $productId = _getUrlParameter('pID');

    /**
     * Category ID
     *
     * Holds the category id from the get parameter.
     * @type {object}
     */
    var $categoryId = _getUrlParameter('cID');

    /**
     * Table Row of Updated Product
     *
     * Table row selector of a product, depending on the pID GET parameter.
     * @type {object|jQuery|HTMLElement}
     */
    var $tableRowOfUpdatedProduct = $('tr[data-id=' + $productId + ']');

    /**
     * Table Row of Updated Category
     *
     * Table row selector of a category, depending on the cID GET parameter.
     * @type {object|jQuery|HTMLElement}
     */
    var $tableRowOfUpdatedCategory = $('tr[data-id=' + $categoryId + ']');

    $tableRowOfUpdatedProduct.addClass('recentlyUpdated');
    $tableRowOfUpdatedCategory.addClass('recentlyUpdated');

    /**
     * Disable/Enable the buttons on the bottom button-dropdown
     * dependent on the checkboxes selection
     * @private
     */
    var _toggleMultiActionBtn = function _toggleMultiActionBtn() {
        var $checked = $('tr[data-id] input[type="checkbox"]:checked');
        $('.js-bottom-dropdown button').prop('disabled', !$checked.length);
    };

    /**
     * Prepare Form
     *
     * @param {string} action
     * @return {object | jQuery}
     *
     * @private
     */
    var _$prepareForm = function _$prepareForm(action) {

        /**
         * Build data object for reference
         * @var {object}
         */
        var data = {
            cPath: '',
            url: [_getSourcePath(), 'categories.php', '?action=multi_action'].join(''),
            pageToken: $('input[name="page_token"]:first').attr('value')
        };

        /**
         * Add cPath
         */
        try {
            data.cPath = window.location.href.match(/cPath=(.*)/)[1];
        } catch (e) {
            data.cPath = $('[data-cpath]:first').data().cpath;
        }
        data.url += '&cPath=' + data.cPath;

        var search = _getUrlParameter('search');
        if (search !== 0 && search !== null) {
            data.url += '&search=' + search;
        }

        /**
         * Build cached form and return it
         * @type {object | jQuery}
         */
        var $form = $('<form name="multi_action_form" method="post" action=' + data.url + '></form>');
        $form.append('<input type="hidden" name="cpath" value=' + data.cPath + '>');
        $form.append('<input type="hidden" name="page_token" value=' + data.pageToken + '>');
        $form.append('<input type="hidden" name=' + action + ' value="Action">');
        $form.appendTo('body');
        return $form;
    };

    /**
     * Map actions for every row in the table.
     *
     * This method will map the actions for each
     * row of the table.
     *
     * @private
     */
    var _mapRowActions = function _mapRowActions() {

        $('.gx-categories-table tr').not('.dataTableHeadingRow').each(function () {

            /**
             * Save that "this" scope here
             *
             * @var {object | jQuery}
             */
            var $that = $(this);

            /**
             * Data attributes of current row
             *
             * @var {object}
             */
            var data = $that.data();

            /**
             * Reference to the row action dropdown
             *
             * @var {object | jQuery}
             */
            var $dropdown = $that.find('.js-button-dropdown');

            /**
             * Fix checkbox event handling conflict and (de-)activate the bottom button-dropdown
             * on checkbox changes
             */
            window.setTimeout(function () {
                $that.find('.single-checkbox').on('click', function (event) {
                    event.stopPropagation();
                    _toggleMultiActionBtn();
                });
            }, 500);

            /**
             * Call action binder method
             */
            if (data.isProduct) {
                _mapProductActions($dropdown, data);
            } else {
                _mapCategoryActions($dropdown, data);
            }

            // Bind icon actions
            // -----------------

            // Open Product / Category
            $that.find('.fa-folder-open-o, .fa-pencil').parent().on('click', function (event) {
                event.preventDefault();
                var url = $that.find('td:eq(2) a[href]:first').prop('href');
                window.open(url, '_self');
            });

            // Delete Product / Category
            $that.find('.fa-trash-o').parent().on('click', function (event) {
                var $deleteItem = $dropdown.find('span:contains(' + jse.core.lang.translate('delete', 'buttons') + ')');
                $deleteItem.click();
            });
        });
    };

    /**
     * Get path of the admin folder
     * Only used start to get the source path
     *
     * @returns {string}
     */
    var _getSourcePath = function _getSourcePath() {
        var url = window.location.origin,
            path = window.location.pathname;

        var splittedPath = path.split('/');
        splittedPath.pop();

        var joinedPath = splittedPath.join('/');

        return url + joinedPath + '/';
    };

    /**
     * Bind an action of a product button to the dropdown.
     *
     * @param action
     * @param $dropdown
     * @param data
     *
     * @private
     */
    var _mapProductAction = function _mapProductAction(action, $dropdown, data) {
        var section = _productSectionNameMapping[action],
            callback = function callback(event) {
            _productConfigurationKeyCallbacks(action, $(event.target), data);
        };
        jse.libs.button_dropdown.mapAction($dropdown, action, section, callback);
    };

    /**
     * Bind an action of a category button to the dropdown.
     *
     * @param action
     * @param $dropdown
     * @param data
     *
     * @private
     */
    var _mapCategoryAction = function _mapCategoryAction(action, $dropdown, data) {
        var section = _categorySectionNameMapping[action],
            callback = function callback(event) {
            _categoryConfigurationKeyCallbacks(action, $(event.target), data);
        };
        jse.libs.button_dropdown.mapAction($dropdown, action, section, callback);
    };

    var _productSectionNameMapping = {
        edit: 'buttons',
        delete: 'buttons',
        BUTTON_MOVE: 'admin_buttons',
        BUTTON_COPY: 'admin_buttons',
        BUTTON_PROPERTIES: 'admin_buttons',
        BUTTON_EDIT_CROSS_SELLING: 'categories',
        GM_BUTTON_ADD_SPECIAL: 'gm_general',
        BUTTON_EDIT_ATTRIBUTES: 'admin_buttons'
    };

    var _categorySectionNameMapping = {
        edit: 'buttons',
        delete: 'buttons',
        BUTTON_MOVE: 'admin_buttons',
        BUTTON_COPY: 'admin_buttons',
        BUTTON_GOOGLE_CATEGORIES: 'categories'
    };

    /**
     * Mapping callback functions of product actions.
     *
     * @param key
     * @param $dropdown
     * @param data
     *
     * @private
     */
    var _productConfigurationKeyCallbacks = function _productConfigurationKeyCallbacks(key, $dropdown, data) {
        switch (key) {
            case 'edit':
                _productEditCallback(data);
                break;
            case 'delete':
                _productDeleteCallback($dropdown);
                break;
            case 'BUTTON_MOVE':
                _productMoveCallback($dropdown);
                break;
            case 'BUTTON_COPY':
                _productCopyCallback($dropdown);
                break;
            case 'BUTTON_PROPERTIES':
                _productPropertiesCallback(data);
                break;
            case 'BUTTON_EDIT_CROSS_SELLING':
                _productEditCrossSellingCallback(data);
                break;
            case 'GM_BUTTON_ADD_SPECIAL':
                _productAddSpecialCallback(data);
                break;
            case 'BUTTON_EDIT_ATTRIBUTES':
                _productEditAttributesCallback(data);
                break;
            default:
                console.alert('Callback not found');
                break;
        }
    };
    /**
     * Execute edit button callback.
     *
     * @private
     */
    var _productEditCallback = function _productEditCallback(data) {
        var url = [_getSourcePath(), 'categories.php', '?pID=' + data.id, '&cPath=' + data.cpath, '&action=new_product', '&page=' + data.page].join('');
        window.open(url, '_self');
    };

    /**
     * Execute delete button callback.
     *
     * @param $dropdown
     *
     * @private
     */
    var _productDeleteCallback = function _productDeleteCallback($dropdown) {
        // Uncheck all checkboxes
        $('.gx-categories-table').find('input[type="checkbox"]').prop('checked', false);

        // Check current checkbox
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').prop('checked', true);

        // Create cached form
        var $form = _$prepareForm('multi_delete');

        // Add checkbox to form
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').clone().appendTo($form);

        // Submit form
        $form.submit();
    };

    /**
     * Execute move button callback.
     *
     * @param $dropdown
     *
     * @private
     */
    var _productMoveCallback = function _productMoveCallback($dropdown) {
        // Uncheck all checkboxes
        $('.gx-categories-table').find('input[type="checkbox"]').prop('checked', false);

        // Check current checkbox
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').prop('checked', true);

        // Create cached form
        var $form = _$prepareForm('multi_move');

        // Add checkbox to form
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').clone().appendTo($form);

        // Submit form
        $form.submit();
    };

    /**
     * Execute copy button callback.
     *
     * @param $dropdown
     *
     * @private
     */
    var _productCopyCallback = function _productCopyCallback($dropdown) {
        // Uncheck all checkboxes
        $('.gx-categories-table').find('input[type="checkbox"]').prop('checked', false);

        // Check current checkbox
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').prop('checked', true);

        // Create cached form
        var $form = _$prepareForm('multi_copy');

        // Add checkbox to form
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').clone().appendTo($form);

        // Submit form
        $form.submit();
    };

    /**
     * Execute property button callback.
     *
     * @private
     */
    var _productPropertiesCallback = function _productPropertiesCallback(data) {
        var url = [_getSourcePath(), 'properties_combis.php', '?products_id=' + data.id, '&cPath=' + data.cpath, '&action=edit_category'].join('');
        window.open(url, '_self');
    };

    /**
     * Execute edit cross selling button callback.
     *
     * @private
     */
    var _productEditCrossSellingCallback = function _productEditCrossSellingCallback(data) {
        var url = [_getSourcePath(), 'categories.php', '?current_product_id=' + data.id, '&cPath=' + data.cpath, '&action=edit_crossselling'].join('');
        window.open(url, '_self');
    };

    /**
     * Execute add special button callback.
     *
     * @private
     */
    var _productAddSpecialCallback = function _productAddSpecialCallback(data) {
        var url = [_getSourcePath(), 'specials.php', '?pID=' + data.id, '&action=' + (data.specialId !== undefined ? 'edit' : 'new'), data.specialId !== undefined ? '&sID=' + data.specialId : ''].join('');
        window.open(url, '_self');
    };

    var _productEditAttributesCallback = function _productEditAttributesCallback(data) {
        var $form = $('<form method="post" action=' + (_getSourcePath() + 'new_attributes.php') + '></form>');
        $form.append('<input type="hidden" name="action" value="edit">');
        $form.append('<input type="hidden" name="current_product_id" value=' + data.id + '>');
        $form.append('<input type="hidden" name="cpath" value=' + data.cpath + '>');
        $form.appendTo('body');
        $form.submit();
    };

    /**
     * Mapping callback functions of category actions.
     *
     * @param key
     * @param $dropdown
     * @param data
     *
     * @private
     */
    var _categoryConfigurationKeyCallbacks = function _categoryConfigurationKeyCallbacks(key, $dropdown, data) {
        switch (key) {
            case 'edit':
                _categoryEditCallback(data);
                break;
            case 'delete':
                _categoryDeleteCallback($dropdown);
                break;
            case 'BUTTON_MOVE':
                _categoryMoveCallback($dropdown);
                break;
            case 'BUTTON_COPY':
                _categoryCopyCallback($dropdown);
                break;
            case 'BUTTON_GOOGLE_CATEGORIES':
                _categoryGoogleCategoriesCallback(data);
                break;
            default:
                console.alert('Callback not found');
                break;
        }
    };
    /**
     * Execute edit button callback.
     *
     * @private
     */
    var _categoryEditCallback = function _categoryEditCallback(data) {
        var url = [_getSourcePath(), 'categories.php', '?cID=' + data.id, '&cPath=' + data.cpath, '&action=edit_category'].join('');
        window.open(url, '_self');
    };

    /**
     * Execute delete button callback.
     *
     * @param $dropdown
     *
     * @private
     */
    var _categoryDeleteCallback = function _categoryDeleteCallback($dropdown) {
        // Uncheck all checkboxes
        $('.gx-categories-table').find('input[type="checkbox"]').prop('checked', false);

        // Check current checkbox
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').prop('checked', true);

        // Create cached form
        var $form = _$prepareForm('multi_delete');

        // Add checkbox to form
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').clone().appendTo($form);

        // Submit form
        $form.submit();
    };

    /**
     * Execute move button callback.
     *
     * @param $dropdown
     *
     * @private
     */
    var _categoryMoveCallback = function _categoryMoveCallback($dropdown) {
        // Uncheck all checkboxes
        $('.gx-categories-table').find('input[type="checkbox"]').prop('checked', false);

        // Check current checkbox
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').prop('checked', true);

        // Create cached form
        var $form = _$prepareForm('multi_move');

        // Add checkbox to form
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').clone().appendTo($form);

        // Submit form
        $form.submit();
    };

    /**
     * Execute copy button callback.
     *
     * @param $dropdown
     *
     * @private
     */
    var _categoryCopyCallback = function _categoryCopyCallback($dropdown) {
        // Uncheck all checkboxes
        $('.gx-categories-table').find('input[type="checkbox"]').prop('checked', false);

        // Check current checkbox
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').prop('checked', true);

        // Create cached form
        var $form = _$prepareForm('multi_copy');

        // Add checkbox to form
        $dropdown.parents('tr:first').find('td:first input[type="checkbox"]').clone().appendTo($form);

        // Submit form
        $form.submit();
    };

    /**
     * Execute google categories callback button.
     *
     * @param data
     *
     * @private
     */
    var _categoryGoogleCategoriesCallback = function _categoryGoogleCategoriesCallback(data) {
        var $lightbox = $('.lightbox_google_admin_categories');
        $lightbox.attr('href', 'google_admin_categories.html?categories_id=' + data.id);
        $lightbox.click();
    };

    /**
     * Map actions for the article dropdown
     *
     * @param params {object}
     *
     * @private
     */
    var _mapProductActions = function _mapProductActions($dropdown, data) {
        _mapProductAction('edit', $dropdown, data);
        _mapProductAction('delete', $dropdown, data); //Bind: Delete (Single Row)
        _mapProductAction('BUTTON_MOVE', $dropdown, data); // Bind: Move
        _mapProductAction('BUTTON_COPY', $dropdown, data); // Bind: Copy
        jse.libs.button_dropdown.addSeparator($dropdown, true); // add a separator to dropdown
        _mapProductAction('BUTTON_PROPERTIES', $dropdown, data); // Bind: Properties
        _mapProductAction('BUTTON_EDIT_CROSS_SELLING', $dropdown, data); // Bind: Cross Selling
        _mapProductAction('GM_BUTTON_ADD_SPECIAL', $dropdown, data); // Bind: New Offer

        if ($dropdown.attr('data-attributes-module-installed') === 'true') {
            _mapProductAction('BUTTON_EDIT_ATTRIBUTES', $dropdown, data); // Bind: edit attributes
        }
    };

    /**
     * Map actions for the category dropdown
     *
     * @param params
     *
     * @private
     */
    var _mapCategoryActions = function _mapCategoryActions($dropdown, data) {
        _mapCategoryAction('edit', $dropdown, data);
        _mapCategoryAction('delete', $dropdown, data); // Bind: Delete
        _mapCategoryAction('BUTTON_MOVE', $dropdown, data); // Bind: Move
        _mapCategoryAction('BUTTON_COPY', $dropdown, data); // Bind: Copy
        jse.libs.button_dropdown.addSeparator($dropdown, true); // add a separator to dropdown
        _mapCategoryAction('BUTTON_GOOGLE_CATEGORIES', $dropdown, data); // Bind: Google categories
    };

    var _selectAllCheckboxes = function _selectAllCheckboxes(event) {
        if ($(event.target).prop('checked') === true) {
            $('input.checkbox').parent().addClass('checked');
            $('input.checkbox').prop('checked', true);
        } else {
            $('input.checkbox').parent().removeClass('checked');
            $('input.checkbox').prop('checked', false);
        }
        _toggleMultiActionBtn();
    };

    var _onMouseEnterStockWarn = function _onMouseEnterStockWarn() {
        $(this).data('shortStockString', $(this).text()); // backup current string
        $(this).text($(this).data('completeStockString')); // display complete string
    };

    var _onMouseLeaveStockWarn = function _onMouseLeaveStockWarn() {
        $(this).text($(this).data('shortStockString'));
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------
    module.init = function (done) {
        // Wait until the buttons are converted to dropdown for every row.
        var interval = setInterval(function () {
            if ($('.js-button-dropdown').length > 0) {
                clearInterval(interval);
                _mapRowActions();

                // Init checkbox checked
                _toggleMultiActionBtn();
            }
        }, 200);

        // Check for selected checkboxes also
        // before all rows and their dropdown widgets have been initialized.
        _toggleMultiActionBtn();

        $('#gm_check').on('click', _selectAllCheckboxes);
        $this.on('mouseenter', '.stock_warn', _onMouseEnterStockWarn).on('mouseleave', '.stock_warn', _onMouseLeaveStockWarn);

        // Finish it
        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
